(defmacro APPLET (name args &rest forms)
"Args: NAME ARGS FORMS
Where NAME is the name of the applet, ARGS is a list of the applets arguments, and FORMS is a series of Lisp statements that are the body of the applet.

  APPLET is similar to DEFUN: Both define code to be evaluted by ViSta.  However, the APPLET is evaluated when ViSta has nothing else to do rather than at a specific moment in a sequence of actions. That is, an applet is evaluated at the first moment that ViSta is idle after the applet has been defined.

  NAME is the symbolic name of the applet.  By convention, the name starts with the @ character.

  ARGS is a list of arguments. In addition to whatever arguments you wish to include, you may take advantage of built-in, pre-defined keyword arguments which control the display of ViSta's DeskTop, XLispStat and (at startup only) Logo Window. These arguments are: Hide-Desktop, Hide-Logo, and Hide-XLispStat (and their anti-aliases Show-DeskTop, Show-Logo and Show-XLispStat). The defaults of these arguments are the values of the global system variables *hide-desktop*, *hide-logo*, and *hide-xlispstat*. The default values for these global variables cause the DeskTop and Logo windows to be seen at startup, while the XLispStat window is not shown.

  FORMS is any series of Lisp statements."

  (let* ((real-args args)
         (args (combine args))
         (loc-hide-desk  (position 'hide-desktop   args))
         (loc-show-desk  (position 'show-desktop   args))
         (loc-hide-logo  (position 'hide-logo      args))
         (loc-show-logo  (position 'show-logo      args))
         (loc-hide-xlsp  (position 'hide-xlispstat args))
         (loc-show-xlsp  (position 'show-xlispstat args))
         (loc-hide-dash  (position 'hide-datasheet args))
         (loc-show-dash  (position 'show-datasheet args))
         (loc-verbose    (position 'verbose        args))
         (hide-desk (cond 
                      (loc-hide-desk      (select args (1+ loc-hide-desk)))
                      (loc-show-desk (not (select args (1+ loc-show-desk))))
                      (t *hide-desktop*)))
         (hide-logo (cond 
                      (loc-hide-logo      (select args (1+ loc-hide-logo)))
                      (loc-show-logo (not (select args (1+ loc-show-logo))))
                      (t *hide-vista*)))
         (hide-dash (cond
                      (loc-hide-dash      (select args (1+ loc-hide-dash)))
                      (loc-show-dash (not (select args (1+ loc-show-dash))))
                      (t *hide-xlispstat*)))
         (hide-xlsp (cond
                      (loc-hide-xlsp      (select args (1+ loc-hide-xlsp)))
                      (loc-show-xlsp (not (select args (1+ loc-show-xlsp))))
                      (t *hide-xlispstat*)))
         (verbose   (if loc-verbose   (select args (1+ loc-verbose))   nil))
         (indices)
         )
    (setf args-now (list loc-hide-desk loc-show-desk loc-hide-logo loc-show-logo
                         loc-hide-dash loc-show-dash loc-hide-xlsp loc-show-xlsp
                         loc-verbose))
    (setf indices  (select args-now (which args-now)))
    (setf indices  (combine (mapcar #'(lambda (index)
                                       (list index (1+ index)))
                                   indices)))
    (setf args-now (select args indices))
    (setf app-args (reverse (set-difference args-now args)))
    (setf app-args (let ((knt 0))
     (remove 'dkjkljksl  (mapcar #'(lambda (arg)
               (if (listp arg)
                   (cond 
                     ((not (member (select args-now knt) arg))
                      `,arg)
                     (t  (setf knt (+ knt 2))
                         'dkjkljksl))
                   `,arg))
           real-args))))
    (send *startup-manager* :hide-desktop   hide-desk)
    (send *startup-manager* :hide-logo      hide-logo)
    (send *startup-manager* :show-xlispstat hide-xlsp)
    (send *startup-manager* :verbose        verbose)

    (setf *hide-desktop*    hide-desk)
    (setf *hide-vista*      hide-desk)
    (setf *hide-logo*       hide-logo)
    (setf *hide-xlispstat*  hide-xlsp)
    (setf *show-xlispstat*  (not hide-xlsp))

    (send *startup-manager* :applet  
          (eval (append `(defun ,name ,app-args) forms)))
    
    (set (intern (string-upcase name)) 
         (send *startup-manager* :applet))
    ))

(defun stop-startup ()
  (stop-all-plots)
  (format t "[ startup stopped ]~%")
  (force-output)
  (top-level)
  (top-level))

(defun startup 
  (&key (startup #'default-system-startup-process) 
        (action nil)
        (applet nil)
        (data   nil)
        (hide-desktop *hide-desktop* hd?)      (show-desktop (not *hide-desktop*) sd?)
        (hide-logo *hide-logo* hl?)            (show-logo (not *hide-logo*) sl?)
        (show-xlispstat *show-xlispstat* sxl?) (hide-xlispstat *hide-xlispstat* hxl?)
        (show-datasheet nil ss?)               (hide-datasheet t hs?)
        (verbose nil)
        (title nil))
"Args &key  (startup #'default-system-startup-process) (action nil) (applet nil) (data nil) (hide-desktop *hide-desktop*) (hide-logo *hide-logo*) (show-xlispstat *show-xlispstat) (verbose nil) (title  ViSta Default Startup)
After starting, ViSta initializes, reads in the -f file and runs STARTUP with keyword :STARTUP NIL to initialize the startup process and create *startup-manager*, an instance of the startup manager prototype object. and carries out the startup process defined by the STARTUP keyword. By default the startup process is (default-system-startup-process), which lets you optionally show or hide the logo, desktop and xlispstat windows as they are created during the process (as controlled by the appropriate keyword arguments). When the startup process concludes the startup manager takes the action defined by either the ACTION, APPLET or DATA keywords (only one can be non-nil). The STARTUP and ACTION arguments must have values which are either a function or NIL. The remaining keywords must have values that are either T or NIL. The window control keywords  have default values specified by the values of the global variables *HIDE-DESKTOP*, *HIDE-LOGO* and *HIDE-XLISPSTAT*. "
  (setf title (cond  (action "Action Startup Process") (applet "Applet Startup Process")
                (data "Data Startup Process") (t "Default Startup Process"))) 
  (setf *startup-process-number* (1+ *startup-process-number* ))
  (when verbose (one-button-dialog (format nil "Startup Process ~a ~a"
        *startup-process-number* title)))
  (when (and sd? hd?) (error "cannot use both show-desktop and hide-desktop"))
  (when (and sl? hl?) (error "cannot use both show-logo and hide-logo"))
  (when (and ss? hs?) (error "cannot use both show-datasheet and hide-datasheet"))
  (when (and sxl? hxl?) (error "cannot use both show-xlispstat and hide-xlispstat"))
  (unless hd? (setf hide-desktop (not show-desktop)))
  (unless hl? (setf hide-logo (not show-logo)))
  (unless hs? (setf hide-datasheet (not show-datasheet)))
  (unless sxl? (setf show-xlispstat (not hide-xlispstat)))
  (setf *startup-manager* 
        (send startup-manager-proto 
              :new startup action applet data *startup-process-number*
                   hide-desktop hide-logo hide-datasheet show-xlispstat show-datasheet
                   title verbose))
  *startup-manager*)


(setf *startup-process-number* 0)
(setf *applet-verbose* nil)
(defun start (&rest args)(apply #'startup args))

(defproto startup-manager-proto
  '(action data applet infile startup hide-desktop hide-logo hide-datasheet show-xlispstat started actioned copyrighted proc# verbose datafile appletfile idling start-time timed) ;vdf 
  '() graph-proto)
  
(defmeth startup-manager-proto :isnew 
              (startup action applet data proc# 
               hide-desktop hide-logo hide-datasheet show-xlispstat show-datasheet
               title verbose)
  (let ((startup-manager 
         (call-next-method 2 :show t :size '(200 1) :location (screen-size))))
    (when (> (length (which (list action applet data))) 1) 
          (error "startup function cannot be invoked with more than one of the ACTION, APPLET, and DATA keyword arguments."))
    (send self :hide-desktop hide-desktop)
    (send self :hide-logo  hide-logo )
    (send self :hide-datasheet (not show-datasheet))
    (send self :show-xlispstat show-xlispstat)
    (send self :startup startup)
    (send self :action action)
    (send self :applet applet)
    (send self :data data)
    (send self :proc# proc#)
    (send self :title title)
    (send self :verbose verbose)
    (send self :idle-on t)
    self))


(defmeth startup-manager-proto :do-idle ()
  "Waits until idle and then, if startup function (which is (default-system-startup-process) by default) hasn't been done, does it. When startup function is done, checks for information in the data, applet or action slots. If any one of these is non-nil, it uses do-data, do-applet or do-action method to evaluate the information in the slot. Only one of the data, applet or action slots may be non-nil. The information in the data and applet slots must be a single data or applet function, respectively. The information in the action can be any lisp code. If this is the first use of the startup function, it then leaves the idle method idling so that it can watch *ini-file* for definition of another data or applet file. If this is not the first use of the startup function the idling method for the method is turned off. Returns t or nil depending on whether an action was taken."

  ;CHECK INI FILE FOR INFILE
  (send self :infile (string-downcase (msw-get-profile-string "ViSta" "InFile" *ini-file*)))
  
  ;WHEN EITHER NOT YET STARTED OR THERE IS AN INFILE, DO SOMETHING
  (when (or (not (send self :started)) (not (equal (send self :infile) "no")))
        (let* ((infile  (send self :infile))
               (type    (if infile (pathname-type infile) nil))
               (verbose (send self :verbose))
               (action  (send self :action))
               (title   (send self :title)))
          
          (cond
            ;WHEN NOT YET STARTED ...
            ((not (send self :started))
             ;AND THERE IS AN INFILE...
             (when (not (equal infile "no"))
             
                   (cond 
               
                     ((equal type "vaf")
                      ;LOAD APPLET FILE NOW TO SET STARTUP OPTIONS
                      ;BUT DEFER PROCESSING UNTIL FINISHED STARTING 
                      (send self :title "Applet Startup Process")
                      (when verbose (one-button-dialog 
                                     (format nil "Process ~a ~a APPLET"
                                             *startup-process-number* title)))
                      (send self :load-vaf infile)
                      )

                     ((equal type "vdf")
                      ;DEFER LOADING DATA FILE UNTIL ACTION TIME
                      (send self :title "Default Data Startup Process")
                      (when verbose (one-button-dialog 
                                     (format nil "Process ~a ~a DATA"
                                             *startup-process-number* title))))))
             
              
             (send self :title "Default Startup Process")
             (when verbose (one-button-dialog 
                            (format nil "Process ~a ~a STARTUP"
                                    *startup-process-number* title)))

             (setf *hide-logo*      (send self :hide-logo))    
             (setf *hide-vista*     (send self :hide-desktop))    
             (setf *hide-desktop*   (send self :hide-desktop))  
             (setf *hide-datasheet* (send self :hide-datasheet))
             (setf *show-xlispstat* (send self :show-xlispstat))
             (send self :do-startup)
             (send self :started t)
             (send self :start-time (run-time))
             (if (> (send self :proc#) 1) (send self :idle-on nil))
             )


            (t  
;WE HAVE FOUND AN INFILE, EITHER JUST NOW OR BEFORE FINISHED STARTING
             
             (when verbose (one-button-dialog 
                            (format nil "Process ~a ~a~%ACTION - READING INFILE TYPE ~a~%~a"
                                    *startup-process-number* title type infile)))
             (cond
               
               ;INFILE IS A PROGRAM FILE
               ((and action (or (equal type "lsp") (equal type "vis")))
                (when verbose 
                (one-button-dialog 
                 (format nil "Process ~a ~a LOADING LSP OR VIS PROGRAM FILE"
                         *startup-process-number* title)))
                (send self :load-lsp infile)
                (send self :do-lsp)
                (msw-write-profile-string "ViSta" "InFile" "No" *ini-file*))
               
               ;INFILE IS A DATA FILE
               ((equal type "vdf") ;DATA FILE LOADED NOW
                (when verbose (one-button-dialog 
                               (format nil "Process ~a ~a LOADING VDF DATA FILE"
                                       *startup-process-number* title)))
                (send self :load-vdf infile)
                (send self :do-data)
                (msw-write-profile-string "ViSta" "InFile" "No" *ini-file*))
               ;INFILE IS AN APPLET FILE
               ((equal type "vaf") ;APPLET CODE IS NOW IN ACTION SLOT, SO DO ACTION
                (when verbose (one-button-dialog 
                               (format nil "Process ~a ~a LOADING VAF APPLET FILE"
                                       *startup-process-number* title)))
		(unless (send self :appletfile)
                        (send self :load-vaf infile))
                (send self :do-applet)
                (send self :appletfile nil)
                (msw-write-profile-string "ViSta" "InFile" "No" *ini-file*)))

             (send self :actioned t)
             (send self :infile nil)
            ; (msw-write-profile-string "ViSta" "InFile" "No" *ini-file*)
             (if (> (send self :proc#) 1) (send self :idle-on nil))
             ))
          
          (or (send self :actioned) (send self :started)))))
  


;STARTUP METHODS

(defmeth startup-manager-proto :startup (&optional (afunc nil set))
  "Message args: (&optional afunc)
 Sets or retrieves the startup function."
  (if set (setf (slot-value 'startup) afunc ))
  (slot-value 'startup))

(defmeth startup-manager-proto :do-startup () 
"Does the startup function"
  (funcall (send self :startup)))

(defmeth startup-manager-proto :started (&optional (logical nil set))
  "Message args: (&optional logical)
 Sets or retrieves whether the startup function has been evaluated."
  (if set (setf (slot-value 'started) logical))
  (slot-value 'started))



(defmeth startup-manager-proto :start-time (&optional (start-time nil set))
  "Message args: (&optional logical)
 Sets or retrieves time ViSta started, in elapsed time units."
  (if set (setf (slot-value 'start-time) start-time))
  (slot-value 'start-time))

(defmeth startup-manager-proto :timed (&optional (logical nil set))
  "Message args: (&optional logical)
 Sets or retrieves whether the time mesage has been changed."
  (if set (setf (slot-value 'timed) logical))
  (slot-value 'timed))

;DATA FILE METHODS	

  (defmeth startup-manager-proto :load-vdf (file)
  (let ((start (get-internal-real-time))
        (proc# (send self :proc#))
        (verbose (send self :verbose))
        (title (send self :title))
        (startup-object self))
    (unless *hide-desktop* 
	(when *desktop-container* 
		(send *desktop-container* :front-window)))
    (send self :datafile file)
    (send self :data `(load-data ,file))
    (msw-write-profile-string "ViSta" "ViStaOp" "Yes" *ini-file*)
    file))

(defmeth startup-manager-proto :data  (&optional (datafunc nil set))
  "Message args: (&optional datafunc)
 Sets or retrieves the data function."
  (if set (setf (slot-value 'data) datafunc ))
  (slot-value 'data))

(defmeth startup-manager-proto :do-data () 
"Does the data function"
  (send *listener* :location 
        (first (send *listener* :location)) (second (send *listener* :location)))
  (when (not *hide-desktop*) (send *desktop-container* :front-window))
  (eval (send self :data))
  (send $ :datafile (send self :infile))
  (send $ :datafile (send self :infile)))

(defmeth startup-manager-proto :datafile (&optional (filename nil set))
  "Message args: (&optional filename)
 Sets or retrieves the name of the file containing the data code."
  (if set (setf (slot-value 'infile) filename))
  (slot-value 'infile))

; APPLET METHODS

(defmeth startup-manager-proto :load-vaf (file)
  (let ((start (get-internal-real-time))
        (proc# (send self :proc#))
        (verbose (send self :verbose))
        (title (send self :title)))
    (unless *hide-desktop* 
	(when *desktop-container* 
		(send *desktop-container* :front-window)))
    (send self :appletfile file)
    (load file)
    (msw-write-profile-string "ViSta" "ViStaOp" "Yes" *ini-file*)
    file))

(defmeth startup-manager-proto :applet  (&optional (appletcode nil set))
  "Message args: (&optional appletcode)
 Sets or retrieves the applet code. The applet code is a list of lisp statements."
  (if set (setf (slot-value 'applet) appletcode))
  (slot-value 'applet))

(defmeth startup-manager-proto :do-applet () 
"Does the applet code"
  (funcall `,(eval (send self :applet))))


(defmeth startup-manager-proto :do-applet () 
"Does the applet function"
(send *listener* :location 
        (first (send *listener* :location)) (second (send *listener* :location)))
  (when (not *hide-desktop*)(send *desktop-container* :front-window))
  (eval (send self :applet))
  (send self :appletfile (send self :infile)))

(defmeth startup-manager-proto :do-applet () 
"Does the applet code"
  (funcall (send self :applet)))

(defmeth startup-manager-proto :appletfile (&optional (filename nil set))
  "Message args: (&optional filename)
 Sets or retrieves the name of the file containing the applet code."
  (if set (setf (slot-value 'appletfile) filename))
  (slot-value 'appletfile))

;ACTION METHODS

(defmeth startup-manager-proto :action  (&optional (afunc nil set))
  "Message args: (&optional afunc)
 Sets or retrieves the action code. The action code is a list of lisp statements."
  (if set (setf (slot-value 'action) afunc ))
  (slot-value 'action))

(defmeth startup-manager-proto :do-action ()  
"Evaluates each statement of the action code"
  (map-elements #'eval (send self :action)))

(defmeth startup-manager-proto :actioned (&optional (logical nil set))
  "Message args: (&optional logical)
 Sets or retrieves whether the action code has been evaluated."
  (if set (setf (slot-value 'actioned) logical))
  (slot-value 'actioned))

(defmeth startup-manager-proto :infile (&optional (filename nil set))
  "Message args: (&optional filename)
 Sets or retrieves the name of the file containing the startup code."
  (if set (setf (slot-value 'infile) filename))
  (slot-value 'infile))

(defmeth startup-manager-proto :proc# (&optional (integer nil set))
  "Message args: (&optional integer)
 Sets or retrieves process-number."
  (if set (setf (slot-value 'proc#) integer))
  (slot-value 'proc#))

(defmeth startup-manager-proto :verbose (&optional (logical nil set))
  "Message args: (&optional integer)
 Sets or retrieves the state of the verbose slot."
  (if set (setf (slot-value 'verbose) logical))
  (slot-value 'verbose))

(defmeth startup-manager-proto :idling (&optional (logical nil set))
  "Message args: (&optional integer)
 Sets or retrieves the state of the verbose slot."
  (if set (setf (slot-value 'idling) logical))
  (slot-value 'idling))

(defmeth startup-manager-proto :copyrighted (&optional (logical nil set))
  "Message args: (&optional logical)
 Sets or retrieves whether the copyright has been shown."
  (if set (setf (slot-value 'copyrighted) logical))
  (slot-value 'copyrighted))

(defmeth startup-manager-proto :hide-desktop (&optional (logical nil set))
  "Message args: (&optional afunc)
 Sets or retrieves whether should hide-desktop at startup"
  (if set (setf (slot-value 'hide-desktop ) logical))
  (slot-value 'hide-desktop ))

(defmeth startup-manager-proto :hide-logo (&optional (logical nil set))
  "Message args: (&optional afunc)
 Sets or retrieves whether should hide-logo"
  (if set (setf (slot-value 'hide-logo) logical))
  (slot-value 'hide-logo))

(defmeth startup-manager-proto :hide-datasheet (&optional (logical nil set))
  "Message args: (&optional afunc)
 Sets or retrieves whether should hide-datasheet at startup"
  (if set (setf (slot-value 'hide-datasheet) logical))
  (slot-value 'hide-datasheet))

(defmeth startup-manager-proto :show-xlispstat (&optional (logical nil set))
  "Message args: (&optional afunc)
 Sets or retrieves whether should show-xlispstat at startup"
  (if set (setf (slot-value 'show-xlispstat) logical))
  (slot-value 'show-xlispstat))




(defun default-system-startup-process ()
  ;(one-button-dialog "Default StartUp")
  (when (and *create-vista* (not *vista-exists*))
        (setf *startup-steps-title* "Startup ViSta - PART B: CREATE VISTA")
        (when *startup-verbose* (show-startup "STEP B 0: creating vista~%"))
        (send *logo* :redraw)
        (setf *vista-startup-error*
              (not (ignore-errors (create-vista :hide-vista *hide-vista*
                                                :hide-logo  *hide-logo*))))
        
        ;STEP C 1 - BEGIN TERMINATION OF STARTUP 
        
        (when (and (not *vista-startup-error*) *startup-verbose*)
	      (setf *startup-steps-title* "Startup ViSta - PART C: READ DATA & SHOW REQUESTED WINDOWS")
              (show-startup (strcat "STEP C 1: ~A Finished Creating ViSta "
                      (if *vista-startup-error* "UNSUCCESSFULLY " "Successfully"))))
        (setf *vista-exists* (not *vista-startup-error*))
        (when *hide-logo* (hidemainwindow))
        (unless *hide-logo*
                (when *show-flying-logo* 
                      (send *logo* :disolve :direction 0 :close nil)
                      (setf *hide-logo* t))
                (send *logo* :location 2000 2000)
                ))
   

  (when *vista-exists* 
        (setf *vista-has-been-shown* nil)
        (setf *deskup-time* (run-time))
        (when *startup-verbose* (show-startup "STEP C 2: TRANSFERING STARTUP DATA/CODE"))
        (when *dde-data* (process-dde-data)
              (when *startup-verbose* 
                    (show-startup "STEP C 3: DDE TRANSFER PROCESSED")))
      
        (when *startup-verbose* (show-startup "STEP C 7: LSP or VDF DATAFILE PROCESSED DUE TO INI-FILE STARTFILE FLAG UP"))
        (when (not *hide-xlispstat*)
              (xlispstat-window)
              (when *startup-verbose* 
                    (show-startup "STEP C 8: SHOWING XLISPSTAT")))
        (prepare-show-vista)
;(one-button-dialog (format nil "Prepared to Show ViSta when both T: ~a ~a" (not *hide-vista*) (not *vista-has-been-shown*)))
        (when (and (not *hide-vista*) (not *vista-has-been-shown*))
              (when *startup-verbose* (show-startup "STEP C 9: SHOWING XLISPSTAT"))
              (show-vista)
              (setf *vista-has-been-shown* t)
              (setf *tracelimit* 1)
              ))

  ;SUCCESSFUL STARTUP PROCESS 
  ;SHOULD NOT HAVE CREATED VISTA AND DIDNT
  
  (when (and (not *create-vista*) (not *vista-exists*))
        (when *startup-verbose* (show-startup "STEP C 10: AS REQUESTED: NOT SHOWING VISTA"))
        (setf *vista-startup-error* nil))

  ;TRAP ERROR CONDITION WHEN IN DEVEL MODE
  ;REVEAL ERROR AND DEVELOPMENT SYSTEM
        
  (when (and *vista-startup-error* *devel-mode* )
        (send *logo* :remove)
        (setf *vista-startup-error* nil) 
       ; (setf *startup-steps-title* "Startup ViSta - PART D: STARTUP ERROR")
        (one-button-dialog "STARTUP ERROR in Development Mode")
        (when *startup-verbose* (show-startup (format nil "~a~%~a" *startup-steps-title* "STARTUP ERROR in Development Mode.")))
        (listeners)
        (print "Error: Startup Error in Development Mode")
        (verbosity 4)
        (debug-options)
	(stop-startup)
        )
                    
  ;TRAP ERROR CONDITION WHEN IN USER MODE
  ;SHOW USER NICE MESSAGE
    
  (when (and *vista-startup-error* (not *devel-mode*))
        (send *logo* :remove)
        (setf *vista-startup-error* nil) 
        (setf *startup-steps-title* "Startup ViSta - PART D: STARTUP ERROR")
        ;(when *startup-verbose* (show-startup "STEP E 1U: STARTUP ERROR in Normal Mode. Error Trapped in ViSta.Lsp"))
        (please-wait "STARTUP ERROR: Try Running ViSta again. If problem persists, send email describing name of error and circumstances to bugs@visualstats.org" :title "Error")
        (defmeth *please-wait* :close () 
          (exit))
        )
  ;NORMAL EXIT
  ;SAVE ENVIRONMENT TO PREFS AND GENESYS
          
  (when *startup-verbose* (show-startup "STEP B 21: Close Logo Window, If Open"))
  (setf *vista-startup-error* nil)
  (save-environment)
  (when *startup-verbose* 
        (setf *startup-steps-title* "Startup ViSta - PART F: NORMAL CONCLUSION")
        (show-startup "STEP F 1: VISTA GUI CONSTRUCTED - NORMAL SYSTEM STARTUP")
        (pause 120)
        ;(send *vista-steps-dialog* :remove)
	)
  )

(defun show-vista ()
"Args: &key (none)
Shows the DeskTop. This function includes two different processes for opening the desktop, one prior to and the other after it has been opened for the first time. Prior to *desk-up* time the initial-help and help-panel are also opened, if appropriate. After *desk-up* only the desktop is opened." 
  (cond 
    ((not (boundp '*desk-up*))
     (post-it "DeskTop Not Constructed"))
    ((not *desk-up*)
       (send *desktop-container* :title 
        (format nil "~a - ~a" *vista-name* *copyright-string*))
     
#|
 |   N E X T   S T A T E M E N T    C A U S E S    D E S K T O P     T O    S H O W
 |#

     (send *desktop-container* :show-window)
     (send *desktop-container* :front-window) ;fwy added oct 7 2002
     (send *desktop-container* :active-window);fwy added oct 7 2002
     (send *desktop-sheet* :location 2000 2000)
     (send *desktop-container* :top-most t);makes desktop active
     (send *desktop-container* :top-most nil)
     (APPLY #'SEND *DESKTOP-CONTAINER* :SIZE (SEND *DESKTOP-CONTAINER* :SIZE))
     (setf *vista-has-been-shown* t)
     (setf *viva* nil)
     (send *workmap* :initialize-file-menu nil)
     (initial-presentations *run-number*))
    (t
     (send *listener* :pop-out nil)
     (send *desktop-container* :front-window)
     (send *desktop-container* :resize)
     (send export-data-file-menu-item :enabled nil)
     (send send-data-to-excel-menu-item  :enabled nil)
     (send save-data-menu-item :enabled nil)
     ))
  (setf *desk-up* t)
  (send *analyze-menu* :enabled nil)
  (save-desktop-settings)
  )

(defun prepare-show-vista ()
;(one-button-dialog "preparing to show desktop")
       ;following was outside and before show-vista
     (when *startup-verbose* (show-startup "STEP C 5: PREPARING TO SHOW VISTA"))
     (enable-container *desktop-container*)
     (send *selector* :bottom-most t)
     (send *selector* :bottom-most nil)
     (send *workmap* :top-most t)
     (send *workmap* :bottom-most nil)
     (send *workmap* :has-v-scroll nil)
     (send *workmap* :has-h-scroll nil)
     (send *selector* :show-window)
     (when *startup-verbose* (show-startup "STEP C 6: LAYOUT"))
     (case *desktop-layout-state*
       (3 (maximize-listener)(restore-desktop))
       (2 (maximize-listener))
       (1 (maximize-listener)(maximize-datasheet))
       (0 (maximize-listener)(maximize-workmap)))
     (apply #'send *desktop-container* :frame-location (- (screen-size) '(4 4)))
     (send *desktop-sheet* :location 2000 2000)
     (setf *bottom-line* nil)
     (send (first (last (all-plots))) :close)
     (when *startup-verbose* (format t   "~%; STEP C 7: SEVEN VALUES~%"))
     (when *startup-verbose* (format t "; in first-time show-vista of vista1.lsp~%"))
     (msw-write-profile-string "ViSta" "ViStaOp" "yes" *ini-file*)
     (if (= *desktop-layout-state* 2)
         (send *desktop-container*
               :make-desktop-container-resize-for-maximized-listener)
         (send *desktop-container* :make-desktop-container-resize))
     (initialize-seven-values)
     (use-seven-values)
     (when (< (min (select *seven-values* (iseq 4))) 1)
           (repair-seven-values)
           (save-seven-values))
     (when *full-screen-desktop* 
           (setf *full-screen-desktop* nil)
           (full-screen-desktop))
     (setf *vista-startup* nil)
     (send *selector* :location 2500 2500)
     (send *desktop-sheet* :location 0 0)
     (send *desktop-sheet* :top-most t)
     (send *desktop-sheet* :bottom-most nil)
     (when *startup-verbose* (format t   "~%; STEP C 8: ACTIVATE WINDOW PANES~%"))
     (send *desktop-container* :active-window)  
     (send *desktop-sheet* :top-most t)
     (send *desktop-sheet* :bottom-most nil)
     (send *desktop-sheet* :location 2000 2000)
     (setcd nil)
     (setcds nil)
     (mapcar #'(lambda (item)
                 (send item :enabled nil))
             (list *show-datasheet-window-item*
                   *show-spreadplot-window-item* 
                   *show-report-window-item*))
     (when (or (= *run-number* 1)
               (< (min *seven-values*) 0))
           (repair-seven-values))
     ;following 2 statements make toolbar redraw neatly for first open data 
     (activate *workmap*)
     (send *workmap* :redraw)
     ;following statement fixes bug with ill-defined do-click
     (defmeth *workmap* :do-click (x y m1 m2 &optional ears)
       (send self :the-real-do-click x y m1 m2 ears))
  )

(defun process-dde-data ()
  "processes data coming to ViSta by DDE from Excel or Access (or elsewhere)"
  (when (and *desktop-container* *dde-data*)
        (let ((start (get-internal-real-time))
              (elapsed))
          (format t "; Start:  Starting ViSta with *DDE-Data* File.~%> ")
          (format t "; DDE-XL: Dynamic Data Exchanged from Excel~%> ")
          (enable-container *desktop-container*)
          (eval *dde-data*)   
          (disable-container)
          (msw-write-profile-string "ViSta" "ViStaOp" "yes" *ini-file*)
          (setf elapsed (/ (- (get-internal-real-time) start)
                           INTERNAL-TIME-UNITS-PER-SECOND))
          (format t ";         Transfer Time: ~d seconds.~%" (fuzz elapsed))
          ;(send $ :info)
          (setf *dde-data* nil)
          (send *workmap* :idle-on nil))
  t))